import React from 'react';
import { StepperProps, OrderStatus } from '../types';
import { STEPS } from '../constants';
import { Check } from 'lucide-react';

export const StepperVariantU: React.FC<StepperProps> = ({ status, className = '' }) => {
  const currentStepIndex = STEPS.findIndex((s) => s.key === status);

  return (
    <div className={`w-full ${className}`}>
      {/* 
        Variant U: Luminous Flow
        - "Extra Ordinary" smooth conic gradient rotation for active processing steps.
        - "Order Created" is treated as INSTANTLY DONE (Static Green), never processing.
        - Active step (if not created/final) has "turning in" animation.
      */}
      <div className="flex w-full items-start justify-between relative pt-4">
          
        {/* Background Line */}
        <div className="absolute top-10 left-0 w-full h-[3px] bg-slate-800 rounded-full" />
        
        {/* Progress Line */}
        <div 
          className="absolute top-10 left-0 h-[3px] bg-gradient-to-r from-emerald-600 to-green-400 transition-all duration-700 ease-out rounded-full shadow-[0_0_15px_rgba(52,211,153,0.4)]"
          style={{ width: `${(currentStepIndex / (STEPS.length - 1)) * 100}%` }}
        />

        {STEPS.map((step, index) => {
          const isActive = index === currentStepIndex;
          const isCompleted = index < currentStepIndex;
          
          const isFinalStep = step.key === OrderStatus.COMPLETED;
          const isCreatedStep = step.key === OrderStatus.ORDER_CREATED;

          // LOGIC CHANGE: 
          // If the step is "Order Created" and it is currently active, we treat it visually as COMPLETED immediately.
          // The user will never see "Order Created" spinning. It is always a static checkmark.
          const isVisuallyCompleted = isCompleted || (isActive && isCreatedStep);

          // We only show the "Turning In" animation if it is Active AND NOT the first step (Created) AND NOT the last step (Done).
          const showSpinner = isActive && !isCreatedStep && !isFinalStep;

          return (
            <div 
              key={step.key} 
              className="flex flex-col items-center relative z-10 group"
              style={{ width: `${100 / STEPS.length}%` }}
            >
              {/* Circle Wrapper */}
              <div className="relative flex items-center justify-center w-14 h-14">
                
                {/* Active: Conic Gradient Border Rotation ("Turning In") */}
                {/* Only shows for intermediate processing steps */}
                {showSpinner && (
                    <div className="absolute inset-0 rounded-full bg-[conic-gradient(transparent_0deg,transparent_90deg,#4ade80_360deg)] animate-[spin_1.5s_linear_infinite]" />
                )}
                
                {/* Active: Inner Mask to create border effect */}
                {showSpinner && (
                    <div className="absolute inset-[3px] rounded-full bg-slate-900 z-0" />
                )}

                {/* Final Step Active: Success Burst */}
                {isActive && isFinalStep && (
                    <div className="absolute inset-0 rounded-full bg-green-500 animate-ping opacity-30" />
                )}

                {/* Main Circle */}
                <div 
                  className={`
                    relative z-10 w-10 h-10 rounded-full flex items-center justify-center transition-all duration-500
                    ${isVisuallyCompleted || (isActive && isFinalStep) ? 'bg-green-500 text-white shadow-[0_0_20px_rgba(74,222,128,0.5)] scale-110' : ''}
                    ${showSpinner ? 'bg-slate-900 text-green-400 scale-100' : ''}
                    ${!isActive && !isCompleted ? 'bg-slate-900 border-2 border-slate-700 text-slate-600' : ''}
                  `}
                >
                   {/* Show Checkmark if completed OR if it's the "Order Created" step (even if active) */}
                   {isVisuallyCompleted || (isActive && isFinalStep) ? <Check size={20} strokeWidth={3} /> : <step.icon size={18} />}
                </div>

              </div>

              {/* Label */}
              <div 
                className={`
                   mt-2 text-center transition-all duration-300
                   ${isActive ? 'translate-y-0 opacity-100' : 'translate-y-1'}
                `}
              >
                <p 
                  className={`
                    text-xs font-bold uppercase tracking-wider
                    ${isActive ? 'text-green-400 drop-shadow-[0_0_8px_rgba(74,222,128,0.5)]' : isCompleted ? 'text-green-600' : 'text-slate-600'}
                  `}
                >
                  {step.label}
                </p>
              </div>
            </div>
          );
        })}
      </div>
    </div>
  );
};