import React from 'react';
import { StepperProps, OrderStatus } from '../types';
import { STEPS } from '../constants';
import { Check } from 'lucide-react';

export const StepperVariantT: React.FC<StepperProps> = ({ status, className = '' }) => {
  const currentStepIndex = STEPS.findIndex((s) => s.key === status);

  return (
    <div className={`w-full ${className}`}>
      {/* 
        Variant T: Kinetic Ring (Fixed)
        - Based on Variant R.
        - Fix: The final "Completed" step does NOT spin. It pulses to show success.
      */}
      <div className="flex w-full items-start justify-between relative">
          
        {/* Background Line */}
        <div className="absolute top-6 left-0 w-full h-1 bg-slate-800 rounded-full" />
        
        {/* Progress Line */}
        <div 
          className="absolute top-6 left-0 h-1 bg-green-500 transition-all duration-500 ease-out rounded-full"
          style={{ width: `${(currentStepIndex / (STEPS.length - 1)) * 100}%` }}
        />

        {STEPS.map((step, index) => {
          const isCompleted = index < currentStepIndex;
          const isActive = index === currentStepIndex;
          const isFinalStep = step.key === OrderStatus.COMPLETED;
          
          // Logic Fix: Only spin if active AND NOT the final "Completed" state.
          const showSpinner = isActive && !isFinalStep;

          return (
            <div 
              key={step.key} 
              className="flex flex-col items-center relative z-10"
              style={{ width: `${100 / STEPS.length}%` }}
            >
              {/* Circle Container */}
              <div className="relative flex items-center justify-center w-12 h-12">
                
                {/* 1. Background Circle */}
                <div className={`
                   absolute inset-0 rounded-full border-4 transition-colors duration-300
                   ${isCompleted || (isActive && isFinalStep) ? 'bg-green-500 border-green-500' : 'bg-slate-900 border-slate-800'}
                   ${isActive && !isFinalStep ? 'border-transparent bg-slate-900' : ''} 
                `} />

                {/* 2. Active Spinner Ring ("Turning In") - Hidden on Final Step */}
                {showSpinner && (
                  <div className="absolute inset-0 rounded-full border-4 border-green-500 border-t-transparent border-l-green-500/50 animate-[spin_1s_linear_infinite]" />
                )}

                {/* 3. Success Pulse for Final Step */}
                {isActive && isFinalStep && (
                   <div className="absolute inset-0 rounded-full border-4 border-green-400 animate-ping opacity-50" />
                )}

                {/* 4. Icon */}
                <div className={`relative z-10 transition-colors ${isCompleted || (isActive && isFinalStep) ? 'text-white' : isActive ? 'text-green-500' : 'text-slate-600'}`}>
                   {isCompleted || (isActive && isFinalStep) ? <Check size={20} strokeWidth={3} /> : <step.icon size={20} />}
                </div>

              </div>

              {/* Label */}
              <div className="mt-3 text-center">
                <p 
                  className={`
                    text-xs font-bold uppercase tracking-wider transition-colors duration-300
                    ${isActive ? 'text-green-400' : isCompleted ? 'text-green-600' : 'text-slate-600'}
                  `}
                >
                  {step.label}
                </p>
                {/* Status Text */}
                {isActive && (
                   <span className={`text-[10px] font-mono block mt-1 ${isFinalStep ? 'text-green-400 font-bold' : 'text-green-500/70'}`}>
                      {isFinalStep ? 'Done' : 'Processing'}
                   </span>
                )}
              </div>
            </div>
          );
        })}
      </div>
    </div>
  );
};