import React from 'react';
import { StepperProps } from '../types';
import { STEPS, PROCESSING_STATUSES, ACTION_REQUIRED_STATUSES } from '../constants';
import { Check, ChevronRight } from 'lucide-react';

export const StepperVariantF: React.FC<StepperProps> = ({ status, className = '' }) => {
  const currentStepIndex = STEPS.findIndex((s) => s.key === status);

  return (
    <div className={`w-full ${className}`}>
      {/* 
        Container: A unified track/bar.
        Looks like a solid progress bar but segmented into content.
      */}
      <div className="w-full bg-slate-900 border border-slate-800 rounded-xl overflow-x-auto no-scrollbar shadow-lg">
        <div className="flex min-w-max">
          {STEPS.map((step, index) => {
            const isCompleted = index < currentStepIndex;
            const isActive = index === currentStepIndex;
            const isProcessing = isActive && PROCESSING_STATUSES.includes(step.key);
            const isActionRequired = isActive && ACTION_REQUIRED_STATUSES.includes(step.key);

            return (
              <div 
                key={step.key} 
                className={`
                  relative flex items-center py-4 px-6 border-r border-slate-800 last:border-r-0 transition-colors duration-300
                  ${isActive ? 'bg-slate-800/80' : 'bg-transparent'}
                  ${isCompleted ? 'bg-slate-900/50' : ''}
                  ${isActionRequired ? '!bg-warning/5' : ''}
                `}
              >
                {/* Top Status Line */}
                <div 
                   className={`
                     absolute top-0 left-0 right-0 h-1 transition-all duration-300
                     ${isCompleted ? 'bg-success' : ''}
                     ${isActive ? 'bg-primary' : 'bg-transparent'}
                     ${isActionRequired ? '!bg-warning' : ''}
                   `}
                />

                {/* Step Content */}
                <div className="flex flex-col gap-1">
                  <div className="flex items-center gap-2">
                     {/* Icon / Indicator */}
                     <div 
                       className={`
                         w-5 h-5 rounded-full flex items-center justify-center text-[10px] font-bold border transition-colors
                         ${isCompleted ? 'bg-success text-white border-success' : ''}
                         ${isActive ? 'bg-primary text-white border-primary' : ''}
                         ${!isActive && !isCompleted ? 'bg-transparent text-slate-600 border-slate-700' : ''}
                         ${isActionRequired ? '!bg-warning !text-black !border-warning' : ''}
                       `}
                     >
                       {isCompleted ? <Check size={12} strokeWidth={3} /> : index + 1}
                     </div>

                     <span 
                       className={`
                         text-sm font-bold whitespace-nowrap
                         ${isCompleted ? 'text-success' : ''}
                         ${isActive ? 'text-white' : ''}
                         ${!isActive && !isCompleted ? 'text-slate-500' : ''}
                         ${isActionRequired ? '!text-warning' : ''}
                       `}
                     >
                       {step.label}
                     </span>
                  </div>
                  
                  {/* Subtext */}
                  <span className={`text-[10px] pl-7 whitespace-nowrap ${isActive ? 'text-slate-400' : 'text-slate-600'}`}>
                    {isActive 
                      ? (isActionRequired ? 'Action Required' : isProcessing ? 'Processing...' : 'Current Step') 
                      : isCompleted 
                         ? 'Done' 
                         : 'Pending'}
                  </span>
                </div>

                {/* Chevron Divider (Visual only) */}
                {index !== STEPS.length - 1 && (
                  <ChevronRight className={`ml-6 w-4 h-4 ${isActive ? 'text-slate-600' : 'text-slate-800'}`} />
                )}
              </div>
            );
          })}
        </div>
      </div>
    </div>
  );
};