import React from 'react';
import { StepperProps } from '../types';
import { STEPS, PROCESSING_STATUSES, ACTION_REQUIRED_STATUSES } from '../constants';
import { Check, Loader2 } from 'lucide-react';

export const StepperVariantE: React.FC<StepperProps> = ({ status, className = '' }) => {
  const currentStepIndex = STEPS.findIndex((s) => s.key === status);

  return (
    <div className={`w-full ${className}`}>
      {/* Container: Glassmorphism pill */}
      <div className="w-full bg-slate-950/40 border border-slate-800/50 backdrop-blur-md rounded-full p-2 flex items-center gap-2 overflow-x-auto no-scrollbar shadow-2xl">
        
        {STEPS.map((step, index) => {
          const isCompleted = index < currentStepIndex;
          const isActive = index === currentStepIndex;
          const isProcessing = isActive && PROCESSING_STATUSES.includes(step.key);
          const isActionRequired = isActive && ACTION_REQUIRED_STATUSES.includes(step.key);
          const isFirstStep = index === 0;

          // Variant E Specific Logic:
          // 1. Active Step: Fully expanded (Icon + Label + Description) + Blinking
          // 2. Completed First Step ("Order Created"): Collapsed to Icon only (to save space)
          // 3. Other Completed Steps: Semi-expanded (Icon + Label) to show history
          // 4. Future Steps: Collapsed (Icon only)

          const showLabel = isActive || (isCompleted && !isFirstStep);
          const showDescription = isActive;

          return (
            <div
              key={step.key}
              className={`
                relative flex items-center rounded-full border transition-all duration-500 ease-[cubic-bezier(0.25,1,0.5,1)] overflow-hidden shrink-0
                ${isActive 
                  ? 'flex-[2_0_auto] bg-slate-900/80 border-slate-700 py-1.5 px-2 pr-6 shadow-inner' 
                  : showLabel
                    ? 'flex-initial bg-slate-800/30 border-transparent py-1.5 px-1.5 pr-4' // Visible history
                    : 'flex-none bg-transparent border-transparent p-1.5' // Collapsed (First step done or Future)
                }
                ${isActionRequired && isActive ? '!border-warning/50 !bg-warning/5' : ''}
              `}
            >
              
              {/* Icon Container */}
              <div 
                className={`
                  relative z-10 w-10 h-10 rounded-full flex items-center justify-center transition-all duration-500
                  ${isCompleted ? 'bg-success/20 text-success' : ''}
                  ${isActive ? 'bg-primary text-white shadow-[0_0_15px_rgba(59,130,246,0.6)] scale-100' : ''}
                  ${!isActive && !isCompleted ? 'bg-slate-800/50 text-slate-600 scale-75' : ''}
                  ${isActionRequired ? '!bg-warning !text-black !shadow-[0_0_15px_rgba(245,158,11,0.6)]' : ''}
                `}
              >
                {/* Active Pulse Effect Ring */}
                {isActive && (
                  <div className={`absolute inset-0 rounded-full border-2 border-white/50 animate-[ping_2s_cubic-bezier(0,0,0.2,1)_infinite]`} />
                )}

                {isCompleted ? (
                   <Check size={18} strokeWidth={3} />
                ) : isProcessing ? (
                   <Loader2 size={18} className="animate-spin" />
                ) : (
                   <step.icon size={18} />
                )}
              </div>

              {/* Text Content */}
              <div 
                className={`
                  flex flex-col justify-center ml-3 transition-all duration-500
                  ${showLabel ? 'opacity-100 translate-x-0' : 'opacity-0 -translate-x-4 max-w-0 overflow-hidden pointer-events-none'}
                  ${isActive ? 'max-w-[200px]' : isCompleted ? 'max-w-[150px]' : ''}
                `}
              >
                <span 
                  className={`
                    font-bold whitespace-nowrap tracking-tight transition-all
                    ${isActive ? 'text-sm text-white' : 'text-xs text-slate-400'}
                    ${isActionRequired ? '!text-warning' : ''}
                    ${isCompleted ? 'line-through decoration-slate-600' : ''}
                  `}
                >
                  {step.label}
                </span>
                
                {/* Description only shows for Active step */}
                <div 
                   className={`
                     overflow-hidden transition-all duration-500
                     ${showDescription ? 'max-h-10 opacity-100 mt-0.5' : 'max-h-0 opacity-0'}
                   `}
                >
                   <span className="text-[10px] text-slate-400 font-medium whitespace-nowrap uppercase tracking-wider leading-none block">
                      {isActionRequired ? 'Action Required' : isProcessing ? 'Processing...' : step.description}
                   </span>
                </div>
              </div>
              
              {/* Background Glow for Active */}
              {isActive && (
                 <div className={`absolute inset-0 opacity-20 bg-gradient-to-r ${isActionRequired ? 'from-warning/50' : 'from-primary/50'} to-transparent blur-md -z-0 pointer-events-none`} />
              )}
            </div>
          );
        })}
      </div>
    </div>
  );
};