import React from 'react';
import { StepperProps } from '../types';
import { STEPS, PROCESSING_STATUSES, ACTION_REQUIRED_STATUSES } from '../constants';
import { Check, Loader2 } from 'lucide-react';

export const StepperVariantD: React.FC<StepperProps> = ({ status, className = '' }) => {
  const currentStepIndex = STEPS.findIndex((s) => s.key === status);

  return (
    <div className={`w-full ${className}`}>
      {/* 
        Container: Glassmorphism pill style. 
        Using backdrop-blur and semi-transparent borders for a premium feel.
      */}
      <div className="w-full bg-slate-950/40 border border-slate-800/50 backdrop-blur-md rounded-full p-2 flex items-center gap-2 overflow-x-auto no-scrollbar shadow-2xl">
        
        {STEPS.map((step, index) => {
          const isCompleted = index < currentStepIndex;
          const isActive = index === currentStepIndex;
          const isProcessing = isActive && PROCESSING_STATUSES.includes(step.key);
          const isActionRequired = isActive && ACTION_REQUIRED_STATUSES.includes(step.key);

          // We use flex-basis and flex-grow to handle the expansion animation
          // Active step gets more space to show the label
          return (
            <div
              key={step.key}
              className={`
                relative flex items-center rounded-full border transition-all duration-500 ease-[cubic-bezier(0.25,1,0.5,1)] overflow-hidden shrink-0
                ${isActive 
                  ? 'flex-[2_0_auto] bg-slate-900/80 border-slate-700 py-1.5 px-2 pr-6 shadow-[inset_0_1px_0_0_rgba(255,255,255,0.1)]' 
                  : 'flex-none bg-transparent border-transparent p-1.5'
                }
                ${isActionRequired && isActive ? '!border-warning/50 !bg-warning/5' : ''}
              `}
            >
              
              {/* Icon Container */}
              <div 
                className={`
                  relative z-10 w-10 h-10 rounded-full flex items-center justify-center transition-all duration-500
                  ${isCompleted ? 'bg-success text-white scale-90' : ''}
                  ${isActive ? 'bg-primary text-white shadow-[0_0_12px_rgba(59,130,246,0.6)] scale-100' : ''}
                  ${!isActive && !isCompleted ? 'bg-slate-800/50 text-slate-500 scale-75' : ''}
                  ${isActionRequired ? '!bg-warning !text-black !shadow-[0_0_12px_rgba(245,158,11,0.6)]' : ''}
                `}
              >
                {isCompleted ? (
                   <Check size={18} strokeWidth={3} />
                ) : isProcessing ? (
                   <Loader2 size={18} className="animate-spin" />
                ) : (
                   <step.icon size={18} />
                )}
              </div>

              {/* Text Content - Reveal only when active */}
              {/* We animate max-width and opacity for a smooth reveal effect */}
              <div 
                className={`
                  flex flex-col justify-center ml-3 transition-all duration-500 delay-75
                  ${isActive ? 'opacity-100 max-w-[200px] translate-x-0' : 'opacity-0 max-w-0 -translate-x-4 pointer-events-none'}
                `}
              >
                <span 
                  className={`
                    text-sm font-bold whitespace-nowrap tracking-tight
                    ${isActionRequired ? 'text-warning' : 'text-white'}
                  `}
                >
                  {step.label}
                </span>
                <span className="text-[10px] text-slate-400 font-medium whitespace-nowrap uppercase tracking-wider leading-none mt-0.5">
                   {isActionRequired ? 'Action Required' : isProcessing ? 'Processing...' : step.description}
                </span>
              </div>
              
              {/* Subtle background glow for active step */}
              {isActive && (
                 <div className={`absolute inset-0 opacity-20 bg-gradient-to-r ${isActionRequired ? 'from-warning/50' : 'from-primary/50'} to-transparent blur-md -z-0 pointer-events-none`} />
              )}
            </div>
          );
        })}
      </div>
    </div>
  );
};