import React from 'react';
import { ChevronRight, CheckCircle2, Clock, AlertCircle } from 'lucide-react';
import { StepperProps } from '../types';
import { STEPS, PROCESSING_STATUSES, ACTION_REQUIRED_STATUSES } from '../constants';

export const StepperVariantC: React.FC<StepperProps> = ({ status, className = '' }) => {
  const currentStepIndex = STEPS.findIndex((s) => s.key === status);

  return (
    <div className={`w-full ${className}`}>
      {/* Desktop: Grid / Horizontal | Mobile: Stack / Vertical */}
      <div className="grid grid-cols-1 md:grid-cols-6 gap-2">
        {STEPS.map((step, index) => {
          const isCompleted = index < currentStepIndex;
          const isActive = index === currentStepIndex;
          const isUpcoming = index > currentStepIndex;
          const isProcessing = isActive && PROCESSING_STATUSES.includes(step.key);
          const isActionRequired = isActive && ACTION_REQUIRED_STATUSES.includes(step.key);

          return (
            <div key={step.key} className="relative group flex md:block">
              
              {/* Card Container */}
              <div 
                className={`
                  flex-1 flex flex-row md:flex-col items-center md:items-start p-3 md:p-4 rounded-xl border transition-all duration-300
                  ${isCompleted 
                    ? 'bg-surface/30 border-success/20 text-slate-400' 
                    : isActive 
                      ? 'bg-surface border-primary shadow-[0_4px_20px_-8px_rgba(59,130,246,0.3)] scale-[1.02] z-10' 
                      : 'bg-transparent border-slate-800 text-slate-600'}
                  ${isActionRequired ? '!border-warning/50 !bg-warning/10' : ''}
                `}
              >
                {/* Header / Icon Row */}
                <div className="flex items-center justify-between w-full mb-0 md:mb-3 mr-4 md:mr-0">
                  <div 
                    className={`
                      p-2 rounded-lg transition-colors
                      ${isCompleted ? 'bg-success/10 text-success' : ''}
                      ${isActive ? 'bg-primary text-white' : ''}
                      ${isUpcoming ? 'bg-slate-800 text-slate-600' : ''}
                      ${isActionRequired ? '!bg-warning !text-black' : ''}
                    `}
                  >
                     {/* Show spinner for processing, check for done, step icon for others */}
                     {isProcessing ? (
                       <step.icon size={18} className="animate-spin" />
                     ) : isCompleted ? (
                       <CheckCircle2 size={18} />
                     ) : isActionRequired ? (
                       <AlertCircle size={18} />
                     ) : (
                       <step.icon size={18} />
                     )}
                  </div>
                  
                  {/* Status Badge (Desktop only usually, but good for active) */}
                  {isActive && !isActionRequired && (
                    <div className="hidden md:block w-2 h-2 rounded-full bg-primary animate-pulse shadow-[0_0_8px_currentColor]" />
                  )}
                  {isActionRequired && (
                    <div className="hidden md:block w-2 h-2 rounded-full bg-warning animate-bounce" />
                  )}
                </div>

                {/* Text Content */}
                <div className="flex-1 min-w-0">
                  <h4 
                    className={`
                      text-sm font-bold truncate transition-colors
                      ${isActive ? 'text-white' : ''}
                      ${isCompleted ? 'text-slate-300' : ''}
                      ${isActionRequired ? '!text-warning' : ''}
                    `}
                  >
                    {step.label}
                  </h4>
                  <p className="text-[10px] md:text-xs text-slate-500 truncate hidden sm:block">
                    {isActive && isProcessing ? 'Processing...' : step.description}
                  </p>
                </div>

                {/* Mobile Status Indicator (Right side) */}
                <div className="md:hidden ml-auto flex items-center">
                    {isCompleted && <CheckCircle2 size={16} className="text-success" />}
                    {isProcessing && <div className="w-2 h-2 bg-primary rounded-full animate-pulse" />}
                    {isActionRequired && <AlertCircle size={16} className="text-warning" />}
                </div>

              </div>

              {/* Connector Arrow (Desktop Only) */}
              {index !== STEPS.length - 1 && (
                <div 
                  className={`
                    hidden md:flex absolute top-1/2 -right-3 -translate-y-1/2 z-20 w-6 h-6 items-center justify-center
                    ${isActive ? 'text-primary' : 'text-slate-700'}
                    ${isCompleted ? 'text-success/50' : ''}
                  `}
                >
                  <ChevronRight size={16} />
                </div>
              )}
              
              {/* Connector Line (Mobile Only) */}
              {index !== STEPS.length - 1 && (
                <div className="md:hidden absolute left-6 top-full h-2 w-0.5 bg-slate-800 -ml-[1px] z-0" />
              )}

            </div>
          );
        })}
      </div>
    </div>
  );
};