import React from 'react';
import { StepperProps } from '../types';
import { STEPS, PROCESSING_STATUSES, ACTION_REQUIRED_STATUSES } from '../constants';
import { Check } from 'lucide-react';

export const StepperVariantB: React.FC<StepperProps> = ({ status, className = '' }) => {
  const currentStepIndex = STEPS.findIndex((s) => s.key === status);

  return (
    <div className={`w-full ${className}`}>
      {/* Container - Ensure labels don't get cut off on edges */}
      <div className="px-4 md:px-8">
        <div className="relative pt-2 pb-10">
          
          {/* Background Track */}
          <div className="absolute top-4 left-0 w-full h-0.5 bg-slate-800 rounded-full" />
          
          {/* Active Track */}
          <div 
            className="absolute top-4 left-0 h-0.5 bg-gradient-to-r from-primary/50 to-primary transition-all duration-700 ease-in-out"
            style={{ 
              width: `${(currentStepIndex / (STEPS.length - 1)) * 100}%` 
            }}
          >
            {/* Glowing leading edge */}
            <div className="absolute right-0 top-1/2 -translate-y-1/2 w-2 h-2 bg-primary rounded-full shadow-[0_0_10px_#3b82f6]" />
          </div>

          {/* Steps */}
          <div className="relative flex justify-between w-full">
            {STEPS.map((step, index) => {
              const isCompleted = index < currentStepIndex;
              const isActive = index === currentStepIndex;
              const isProcessing = isActive && PROCESSING_STATUSES.includes(step.key);
              const isActionRequired = isActive && ACTION_REQUIRED_STATUSES.includes(step.key);

              return (
                <div 
                  key={step.key} 
                  className="flex flex-col items-center group cursor-default"
                >
                  {/* Node Dot */}
                  <div 
                    className={`
                      relative w-4 h-4 rounded-full border-2 z-10 transition-all duration-300
                      ${isCompleted ? 'bg-success border-success' : ''}
                      ${isActive ? 'bg-background border-primary scale-125' : ''}
                      ${!isActive && !isCompleted ? 'bg-background border-slate-700' : ''}
                      ${isActionRequired ? '!border-warning !bg-background' : ''}
                    `}
                  >
                    {/* Animated Ping for Active */}
                    {(isActive) && (
                        <div className={`absolute -inset-1 rounded-full opacity-20 animate-ping ${isActionRequired ? 'bg-warning' : 'bg-primary'}`}></div>
                    )}
                  </div>

                  {/* Label - Absolute positioned to center below dot without affecting flex layout width */}
                  <div className="absolute top-8 w-24 flex flex-col items-center">
                    <span 
                      className={`
                        text-xs font-medium text-center transition-all duration-300
                        ${isActive ? 'text-white translate-y-0 opacity-100' : ''}
                        ${isCompleted ? 'text-slate-400 opacity-80' : ''}
                        ${!isActive && !isCompleted ? 'text-slate-600 opacity-60' : ''}
                        ${isActionRequired ? '!text-warning !opacity-100' : ''}
                      `}
                    >
                      {step.label}
                    </span>
                    
                    {/* Minimal status indicator for active step */}
                    {isActive && (
                      <span className={`text-[10px] mt-0.5 ${isActionRequired ? 'text-warning' : 'text-primary/70'}`}>
                         {isActionRequired ? 'Action req.' : isProcessing ? 'Loading...' : 'Current'}
                      </span>
                    )}
                  </div>
                </div>
              );
            })}
          </div>
        </div>
      </div>
    </div>
  );
};