import React from 'react';
import { Check } from 'lucide-react';
import { StepperProps } from '../types';
import { STEPS, PROCESSING_STATUSES, ACTION_REQUIRED_STATUSES } from '../constants';

export const StepperVariantA: React.FC<StepperProps> = ({ status, className = '' }) => {
  const currentStepIndex = STEPS.findIndex((s) => s.key === status);

  return (
    <div className={`w-full ${className}`}>
      {/* Scrollable container for mobile */}
      <div className="flex w-full items-start overflow-x-auto no-scrollbar pb-4 md:pb-0">
        <div className="min-w-[600px] md:min-w-0 w-full flex justify-between relative">
          
          {/* Connecting Line Background */}
          <div className="absolute top-5 left-0 w-full h-1 bg-slate-700 -z-0 rounded-full" />
          
          {/* Active Progress Line (Green for completion) */}
          <div 
            className="absolute top-5 left-0 h-1 bg-success transition-all duration-500 ease-out -z-0 rounded-full"
            style={{ 
              width: `${(currentStepIndex / (STEPS.length - 1)) * 100}%` 
            }}
          />

          {STEPS.map((step, index) => {
            const isCompleted = index < currentStepIndex;
            const isActive = index === currentStepIndex;
            const isProcessing = isActive && PROCESSING_STATUSES.includes(step.key);
            const isActionRequired = isActive && ACTION_REQUIRED_STATUSES.includes(step.key);
            
            return (
              <div 
                key={step.key} 
                className="flex flex-col items-center relative z-10 group"
                style={{ width: `${100 / STEPS.length}%` }}
              >
                {/* Circle Indicator */}
                <div 
                  className={`
                    w-10 h-10 rounded-full flex items-center justify-center border-4 transition-all duration-300
                    ${isCompleted ? 'bg-success border-success text-white' : ''}
                    ${isActive ? 'bg-surface border-primary text-primary scale-110 shadow-[0_0_15px_rgba(59,130,246,0.5)]' : ''}
                    ${!isActive && !isCompleted ? 'bg-surface border-slate-600 text-slate-500' : ''}
                    ${isActionRequired ? '!border-warning !text-warning !shadow-[0_0_15px_rgba(245,158,11,0.5)]' : ''}
                  `}
                >
                  {isCompleted ? (
                    <Check size={18} strokeWidth={3} />
                  ) : (
                    <step.icon size={18} className={isProcessing ? 'animate-pulse' : ''} />
                  )}
                  
                  {/* Processing Ring */}
                  {isProcessing && (
                     <div className="absolute inset-0 rounded-full border-4 border-primary border-t-transparent animate-spin opacity-50"></div>
                  )}
                </div>

                {/* Label */}
                <div className="mt-3 text-center px-1">
                  <p 
                    className={`
                      text-sm font-semibold transition-colors duration-300 whitespace-nowrap
                      ${isActive || isCompleted ? 'text-slate-100' : 'text-slate-500'}
                      ${isActionRequired ? '!text-warning' : ''}
                    `}
                  >
                    {step.label}
                  </p>
                  {/* Status Helper Text */}
                  {(isActive) && (
                     <span className={`text-[10px] font-medium uppercase tracking-wider block mt-1 ${isActionRequired ? 'text-warning' : 'text-primary'}`}>
                        {isActionRequired ? 'Action Needed' : isProcessing ? 'Processing...' : 'Active'}
                     </span>
                  )}
                </div>
              </div>
            );
          })}
        </div>
      </div>
    </div>
  );
};