import React, { useState } from 'react';
import { OrderStatusStepper } from './components/OrderStatusStepper';
import { OrderStatus, Variant } from './types';
import { STEPS } from './constants';
import { ChevronRight, RotateCcw, LayoutTemplate } from 'lucide-react';

const App: React.FC = () => {
  // Default: Start at 'awaiting_deposit' so 'order_created' appears done immediately.
  const [currentStatus, setCurrentStatus] = useState<OrderStatus>(OrderStatus.AWAITING_DEPOSIT);

  const currentIndex = STEPS.findIndex(s => s.key === currentStatus);

  const handleNext = () => {
    if (currentIndex < STEPS.length - 1) {
      setCurrentStatus(STEPS[currentIndex + 1].key);
    }
  };

  const handleReset = () => {
    // Reset to AWAITING_DEPOSIT, because "Order Created" is conceptually always done in the past.
    setCurrentStatus(OrderStatus.AWAITING_DEPOSIT);
  };

  const VariantSection = ({ variant, title, subtitle }: { variant: Variant, title: string, subtitle: string }) => (
    <section>
      <div className="mb-4 flex items-baseline justify-between border-b border-slate-800 pb-2">
        <h2 className="text-lg font-semibold text-white flex items-center gap-2">
          Variant {variant} <span className="text-[10px] bg-slate-800 text-slate-400 px-2 py-0.5 rounded-full uppercase tracking-wider">{title}</span>
        </h2>
        <span className="text-xs font-mono text-slate-500">{subtitle}</span>
      </div>
      <div className="bg-surface/50 border border-slate-800 rounded-xl p-8 shadow-sm">
        <OrderStatusStepper status={currentStatus} variant={variant} />
      </div>
    </section>
  );

  return (
    <div className="min-h-screen w-full bg-background text-slate-200 p-6 md:p-12">
      
      {/* -------------------- Simulator Controls -------------------- */}
      <div className="max-w-4xl mx-auto mb-16 bg-surface border border-slate-700 rounded-xl p-6 shadow-xl sticky top-0 z-50 backdrop-blur-lg bg-surface/90">
        <div className="flex flex-col md:flex-row items-start md:items-center justify-between gap-6">
          <div>
            <h1 className="text-2xl font-bold text-white mb-2 flex items-center gap-2">
              <LayoutTemplate className="text-primary" />
              Order Status Stepper
            </h1>
            <p className="text-slate-400 text-sm">Use the controls to simulate order progress.</p>
          </div>

          <div className="flex flex-wrap items-center gap-3 w-full md:w-auto">
            <select 
              value={currentStatus}
              onChange={(e) => setCurrentStatus(e.target.value as OrderStatus)}
              className="bg-slate-900 border border-slate-700 text-sm rounded-lg focus:ring-primary focus:border-primary block w-48 p-2.5"
            >
              {STEPS.map((step) => (
                <option key={step.key} value={step.key}>{step.label}</option>
              ))}
            </select>

            <button 
              onClick={handleNext}
              disabled={currentIndex === STEPS.length - 1}
              className="flex items-center gap-2 px-4 py-2 bg-primary hover:bg-blue-600 disabled:opacity-50 disabled:cursor-not-allowed text-white rounded-lg font-medium transition-colors text-sm"
            >
              Next Step <ChevronRight size={16} />
            </button>

            <button 
              onClick={handleReset}
              className="p-2 text-slate-400 hover:text-white hover:bg-slate-700 rounded-lg transition-colors"
              title="Reset"
            >
              <RotateCcw size={20} />
            </button>
          </div>
        </div>
      </div>

      {/* -------------------- Variant Demos -------------------- */}
      <div className="max-w-6xl mx-auto grid grid-cols-1 gap-16 pb-20">
        
        {/* Requested Improvements */}
        <div className="space-y-12 border-b border-slate-800 pb-12">
          <h3 className="text-xl font-bold text-primary">Top Pick</h3>
          <VariantSection variant="U" title="Luminous Flow" subtitle="Extra-ordinary: Smooth conic rotation for active step. Order Created is always done." />
        </div>
        
        <div className="space-y-12 border-b border-slate-800 pb-12">
          <h3 className="text-xl font-bold text-slate-400">Other Modern Options</h3>
          <VariantSection variant="T" title="Kinetic Ring v2" subtitle="Stops spinning on Completion, steady green success." />
          <VariantSection variant="R" title="Kinetic Ring (Legacy)" subtitle="Original spinner implementation" />
          <VariantSection variant="Q" title="Exchange Standard" subtitle="Professional, clean, minimal (Binance/Kraken style)" />
          <VariantSection variant="S" title="Orbital Elite" subtitle="Extraordinary motion with orbiting satellite" />
        </div>

        {/* Main Highlights */}
        <div className="space-y-12">
          <h3 className="text-xl font-bold text-slate-400">Highlights</h3>
          <VariantSection variant="E" title="History Horizon" subtitle="Expanding pill, collapses 'Created' on completion" />
          <VariantSection variant="G" title="Neon Cyberpunk" subtitle="Glowing green lines, dark mode optimized" />
          <VariantSection variant="H" title="Square Tech" subtitle="Sharp edges, technical look" />
        </div>

        {/* The New Collection (Variants I - P) */}
        <div className="space-y-12">
          <h3 className="text-xl font-bold text-slate-400">Design Collection</h3>
          <VariantSection variant="I" title="Minimal Dots" subtitle="Clean, focused on process" />
          <VariantSection variant="J" title="Thick Track" subtitle="Inset bar style" />
          <VariantSection variant="K" title="Diamond Nodes" subtitle="Rotated square indicators" />
          <VariantSection variant="L" title="Floating Glass" subtitle="Backdrop blur and transparency" />
          <VariantSection variant="M" title="Underline" subtitle="Bottom progression bar" />
          <VariantSection variant="N" title="Dashed Connector" subtitle="Dashed future path" />
          <VariantSection variant="O" title="Connected Pills" subtitle="Label inside the node" />
          <VariantSection variant="P" title="Inset Neumorphic" subtitle="Soft shadows and depth" />
        </div>

        {/* Classics */}
        <div className="space-y-12 pt-8 border-t border-slate-800">
           <h3 className="text-xl font-bold text-slate-500">Classic Variants</h3>
           <VariantSection variant="A" title="Classic Circles" subtitle="Standard timeline with green progress" />
           <VariantSection variant="F" title="Segmented Track" subtitle="Structured bar" />
           <VariantSection variant="D" title="Original Expanding" subtitle="The predecessor to Variant E" />
        </div>

      </div>
    </div>
  );
};

export default App;